/**
 * Module for utility functions related to file paths and JSON conversion.
 * @module idx-utils
 */

let $$ = require('cdev/debug')({debug: 'cache', path: module.id, prefix: '[isg.xpert.archive]'});
$$.mdb = require('/agorum/roi/customers/cdev.ncore/js/utils/mdb')($$);
$$.pdf = require('/agorum/roi/customers/cdev.ncore/js/utils/pdf')($$);
$$.md = require('/agorum/roi/customers/cdev.ncore/js/utils/md')($$);
$$.objects = require('common/objects');

let ac = require('ac');
// let time = require('common/time');

// load config settings from metadb
const CONFIG = 'MAIN_MODULE_MANAGEMENT/customers/isg.xpert.archive/config';
let config = $$.mdb.load(CONFIG);
let PREFIX = config.prefix;
let IDX_CHARSET = config.charset;

const validatePdfObject = (obj, meta) => {
  let validation = {};

  $$.debug('validatePdfObject', obj, meta);

  let pdDocument = $$.pdf.loadFromObject(obj, {
    error: (object, e) => {
      validation[PREFIX + 'pdfValidation'] = false;
      validation[PREFIX + 'pdfPages'] = null;
      validation[PREFIX + 'pdfError'] = e.message;
      $$.log('ERROR [{0}] for [{1}] at [{2}]'.format(e.message, object.name, meta[PREFIX + 'insNumber']));
    },
    success: (object, pdDocument) => {
      validation[PREFIX + 'pdfValidation'] = true;
      validation[PREFIX + 'pdfPages'] = pdDocument.getNumberOfPages();
      validation[PREFIX + 'pdfError'] = null;
      $$.log('SUCCESS [{0}] at [{1}] validated'.format(object.name, meta[PREFIX + 'insNumber']));
    },
    // use default validation function: isEncrypted, getNumberOfPages
  });
  if (pdDocument) {
    pdDocument.close();
  }
  return Object.assign(meta, validation);
};

/**
 * Converts the content of a stream to a string using the specified character set.
 *
 * @param {java.io.InputStream} stream - The stream to convert.
 * @param {string} charset - The character set to use for the conversion.
 * @returns {string} The content of the stream as a string.
 */
const convertContent = (stream, charset) => {
  let Charset = global.java.nio.charset.Charset;
  let reader = new global.java.io.BufferedReader(
    new global.java.io.InputStreamReader(stream, Charset.forName(charset))
  );
  let line,
    content = [];
  while ((line = reader.readLine()) !== null) {
    content.push(line);
  }
  return content.join('\n');
};

/**
 * Moves images from a source object to one or more target $$.objects.
 *
 * @param {string} _idx - The ID of the source object.
 * @param {Object} meta - The metadata of the source object.
 * @param {string[]|string} _targets - The ID(s) of the target object(s).
 * @param {boolean} save - Whether to save the metadata after moving the images.
 * @throws {Error} If no images are found for the source object or if an image file is not found.
 */
const moveImages = (_idx, meta, _targets, save) => {
  let idx = $$.objects.tryFind(_idx);
  let targets = Array.isArray(_targets) ? _targets : [_targets];
  let images = '{0}images'.format(PREFIX);
  $$.debug('moveImages', idx, meta, targets, save);
  // process imagefiles and silently ignore empty array here
  (meta[images] || []).forEach(imgName => {
    let parent = $$.objects.find(idx.firstParent);
    let img = $$.objects.tryFind(parent.getItem(imgName));
    if (!img) throw new Error('image file [{0}] not found'.format(imgName));
    let metaPdf = $$.md.load(img, new RegExp('^{0}'.format(PREFIX)));
    // validate pdf image file and log error if not already done
    if (!(PREFIX + 'pdfValidation' in metaPdf)) {
      meta = validatePdfObject(img, meta);
    }
    /*
      save && $$.md.save(img, meta);
      if (meta[PREFIX + 'pdfValidation'] === false) {
        throw new Error('PDF validation for [{0}] failed'.format(imgName));
      }
    }
    */
    save && $$.md.save(img, meta);
    $$.objects.unlink(img);
    $$.objects.link(img, targets);
    $$.log('moved [{0}] to [{1}]'.format(img.name, targets.map(t => t.name).join(',')));
  });
};

/**
 * Converts a string of content into a JSON object with specified fields.
 * @param {string} content - The content to be converted to JSON.
 * @param {string} prefix - The prefix to be added to each field in the resulting JSON object.
 * @returns {Object} - The resulting JSON object with mapped fields.
 */
const convertToJSON = (content, prefix) => {

  let idxFormatVersion = null;
  let lastLine = null;
  let result = {};

  const addPrefix = field => {
    return prefix && prefix.length > 0 ? prefix + field : field;
  };

  const lines = content.split('\n'); /* .filter(line => line.trim() !== ''); */

  const mapping = {
    2: 'processCode',
    4: 'orderNumber',
    5: 'insPatId',
    6: 'orderDate',
    7: 'insNumber',
    8: 'processDate',
    9: 'insSurname',
    10: 'insForename',
    11: 'insBirthday',
    12: 'processLvl1', // "Gutachten"
    13: 'processLvl2', // Leistungsbereich
    14: 'processLvl3', // Belegart, documentType (legacy)
    15: 'processFlag', // "AMS"
    16: 'class_code',  // Dokumentenklasse
    17: 'type_code',   // Dokumententyp
    18: 'event_code_list', // Tätigkeitskennzeichen
  };

  result[addPrefix('processCode')] = lines[1].trim();
  let processCode = lines[1].trim();

  if (lines[17].trim() === '</idx>') {
    idxFormatVersion = 'IDXv1';
    lastLine = 14;
  } else if (lines[18].trim() === '</idx>') {
    idxFormatVersion = 'IDXv2';
    lastLine = 17;
  } else {
    idxFormatVersion = null;
    result[addPrefix('processError')] = 'invalid idx format: </idx> not found at line 17 or 18';
    // TODO: wait for intecsoft reply on idx format
    lastLine = 14;
    // return result;
  }

  result[addPrefix('idxFormatVersion')] = idxFormatVersion;

  // parse content text lines and map to fields
  let images = [];
  let imageLine = false;
  for (let i = 0; i < lines.length; i++) {
    if (i <= lastLine) {
      let field = mapping[i + 1]; // Zeilennummern starten bei 1
      // quick and dirty: convert some attributes to date, use string otherwise
      if (field) {
        result[addPrefix(field)] =
          ['orderDate', 'processDate', 'insBirthday'].indexOf(field) >= 0
            ? stringToDate(lines[i].trim(), processCode !== '0')
            : lines[i].trim();
      }
    } else {
      if (imageLine && lines[i].trim() != '</img>') {
        images.push(lines[i].trim());
      } else if (lines[i].trim() === '<img>') {
        imageLine = true;
      } else if (lines[i].trim() === '</img>') {
        imageLine = false;
      }
    }
  }

  if (processCode === '0' && images.length === 0) {
    result[addPrefix('processError')] = 'no images found for [{0}/{1}]'.format(
      result[addPrefix('orderNumber')],
      result[addPrefix('insNumber')]
    );
  }
  result[addPrefix('insName')] = [result[addPrefix('insForename')], result[addPrefix('insSurname')]].join(' ');
  result[addPrefix('images')] = images;
  result[addPrefix('imgName')] = images.length > 0 ? images[0] : null;
  return result;
};

/**
 * Splits a file path into its components.
 * @param {string} path - The file path to be split.
 * @returns {Object} - An object containing the path, file name, base name, name, extension, and type.
 */
const splitPath = path => {
  let file = path.split('/').slice(-1)[0]; // filename without directory path
  let type = file.split('.').slice(-1)[0]; // second extension => type [ ok, idx]
  let name = file.split('.').slice(0, -1).join('.'); // filename without second extension
  let ext = name.split('.').slice(-1)[0]; // first extension => pdf or base (if missing)
  let base = name.split('.').slice(0, -1).join('.'); // filename without first extension
  return { path: path, file: file, base: base, name: name, ext: ext, type: type };
};

/**
 * Generates the archive path for a given record ID.
 * @param {number} recordId - The record ID to generate the archive path for.
 * @param {Object} _opts - Optional parameters for the archive path generation.
 * @param {number} _opts.size - The number of digits per block (default: 3).
 * @param {number} _opts.count - The number of blocks (default: 3).
 * @param {boolean} _opts.trunc - Whether to truncate the archive path to count-1 levels of directories (default: true).
 * @returns {string} - The archive path for the given record ID.
 */
const archivePath = (recordId, _opts) => {
  let opts = Object.assign({ size: 3, count: 3, trunc: true }, _opts);
  let blockTrunc = opts.trunc ? 1 : 0; // count-1 levels of directories
  let blockSize = opts.size; // digits per block => 1000 records per block
  let blockCount = opts.count + blockTrunc; // blocks => 1.000.000.000 records per archive
  let paddedId = recordId.toString().padStart(blockSize * blockCount, '0');
  let blocks = [];
  for (let i = 0; i < blockCount - blockTrunc; i++) {
    blocks.push(paddedId.slice(i * blockSize, (i + 1) * blockSize));
  }
  return blocks.join('/');
};

/**
 * Converts a date string in the format "dd.mm.yyyy" to a Date object.
 * @param {string} dateString - The date string to be converted.
 * @returns {Date} - The resulting Date object.
 */
const stringToDate = (dateString, ignoreError) => {
  // let [day, month, year] = dateString.split('.').map(Number);
  // return new Date(year, month, day);
  // return dateString ? ac.parseDate(dateString, 'dd.MM.yyyy') : null;
  let result = null;
  try {
    result = ac.parseDate(dateString, 'dd.MM.yyyy');
  } catch (e) {
    if (ignoreError) {
      result = null;
    } else {
      throw e;
    }
  }
  return result;
};

/**
 * Converts a Date object to a string in the format "dd.mm.yyyy".
 * @param {Date} date - The Date object to be converted.
 * @returns {string} - The resulting date string.
 */
const dateToString = dateObj => {
  // const day = date.getDate().toString().padStart(2, '0');
  // const month = (date.getMonth() + 1).toString().padStart(2, '0');
  // const year = date.getFullYear();
  // return '{0}.{1}.{2}'.format(day, month, year);
  return ac.formatDate(dateObj, 'dd.MM.yyyy');
};

module.exports = {
  IDX_CHARSET: IDX_CHARSET,
  PREFIX: PREFIX,
  convertContent: convertContent,
  convertToJSON: convertToJSON,
  moveImages: moveImages,
  splitPath: splitPath,
  archivePath: archivePath,
  dateToString: dateToString,
  stringToDate: stringToDate,
  validatePdfObject: validatePdfObject,
};
